unit mysql_udf;

interface

uses SysUtils, windows, contnrs;

{Author: Nicholas Sherlock, 2006
 http://www.sherlocksoftware.org

 Init function prototype:
 function (var init: UDF_INIT; var _args: UDF_ARGS;
    message: pChar): my_bool; cdecl;

 Deinit function prototype:
 procedure Levenshtein_deinit(var init: UDF_INIT); cdecl;

 Your main function's prototype should be like this, for:

 STRING functions:

 function(var initid:UDF_INIT; var args:UDF_ARGS;
          result:PChar; var length:cardinal;
          var is_null, error:byte):PChar;

 For INTEGER functions:

 function(var initid:UDF_INIT; var args:UDF_ARGS;
          var is_null, error:byte):Int64;

 For REAL functions:

 function(var initid:UDF_INIT; var args:UDF_ARGS;
          var is_null, error:byte):Double;

 Save your .DLL to your MySql's Bin folder.

 You can find documentation on MySQL UDFs at:
   http://dev.mysql.com/doc/refman/5.1/en/adding-udf.html
 The documentation is not very coherent so you must read every
 page on UDFs carefully!

 The C++ example UDFs are found in the file Sql/Udf_Example.c in the
 MySQL source distribution.
}

const
  STRING_RESULT = 0;
  REAL_RESULT = 1;
  INT_RESULT = 2;
  ROW_RESULT = 3;
  DECIMAL_RESULT = 4;

  //From mysql_com.h

  MYSQL_ERRMSG_SIZE = 512;

type
  my_bool = byte;
  uint = LongWord;
  ulong = LongWord;

  puint = ^uint; // pointer to an array of uints
  pUlong = ^ulong;

  ItemResult = STRING_RESULT..DECIMAL_RESULT;
  pItemResult = ^Integer;

  pByte = ^byte;
  pDouble = ^Double;

  ItemResultArr = array[0..maxint div sizeof(integer) - 1] of Integer;
  CharArr = array[0..maxint div SizeOf(PChar) - 1] of pChar;
  uintArr = array[0..maxint div sizeof(uint) - 1] of uint;
  ByteArr = array[0..maxint div SizeOf(byte) - 1] of byte;
  Int64Arr = array[0..maxint div sizeof(Int64) - 1] of int64;

  pItemResultArr = ^ItemResultArr;
  pCharArr = ^CharArr;
  puintArr = ^uintArr;
  pByteArr = ^ByteArr;
  pInt64Arr = ^Int64Arr;

type
  pUDF_ARGS = ^UDF_ARGS;
  UDF_ARGS = packed record
    arg_count: uint; // Number of arguments
    arg_type: pItemResultArr; // Array of argument types
    args: pCharArr; // Pointer to argument data
    lengths: pUIntArr; // Length of string arguments
    maybe_null: pByteArr; // Set to 1 for all maybe_null args
    attributes: pCharArr; // Pointer to attribute name
    attribute_lengths: puintArr; // String lengths of attribute arguments
  end;

  {Basically just provides an interpretation of UDF_ARGS in the context of a
   single argument}
  TUDF_Argument = class
  private
    fargs: pUDF_ARGS;
    findex: integer; //Our index in args arrays

    procedure setargtype(value: ItemResult);
    function getArgType: ItemResult;
    function getAttributes: string;
    function getDataString: string;
    function getDataDouble: double;
    function getDataInt: int64;
    function getcanbenull: Boolean;
    procedure setcanbenull(value: boolean);
    function getisnull: boolean;
  public
    constructor create(args: pUDF_ARGS; index: integer);

    property IsNull: boolean read getisnull;
    property CanBeNull: boolean read getcanbenull write setcanbenull;
    property Attributes: string read getAttributes;
    property ArgType: ItemResult read getargtype write setargtype;
    property DataString: string read getDataString;
    property DataDouble: double read getDataDouble;
    property DataInteger: int64 read getDataInt;
  end;

  TUDF_Args = class
  private
    fargs: TObjectList;
    function getargument(index: integer): TUDF_Argument;
  public

    function Count: integer;
    property Argument[index: Integer]: TUDF_Argument read getargument; default;

    constructor create(args: pUDF_ARGS);
    destructor destroy; override;
  end;

// This holds information about the result

  pUDF_INIT = ^UDF_INIT;
  UDF_INIT = packed record
    maybe_null: my_bool; // 1 if function can return NULL
    decimals: uint; // for real functions
    max_length: uint; // For string functions
    ptr: pChar; // free pointer for function data
    const_item: my_bool; // 1 if result is independent of arguments, only available in v5.1!!
  end;

  TUDF_Init = class
  private
    fInit: pUDF_INIT;
    function getCanBeNull: boolean;
    function getData: pointer;
    function getDecimals: Cardinal;
    function getMaxStrLen: cardinal;
    procedure setCanBeNull(const Value: boolean);
    procedure setData(const Value: pointer);
    procedure setDecimals(const Value: Cardinal);
    procedure setMaxStrLen(const Value: cardinal);
  public
    //True if the function can return null
    property CanBeNull: boolean read getCanBeNull write setCanBeNull;
    //True if the result is independent of the arguments.
    property IsConst: boolean read getCanBeNull write setCanBeNull;

    //Maximum length of result string
    property MaxStrLen: cardinal read getMaxStrLen write setMaxStrLen;
    //Maximum number of characters after the decimal point
    property Decimals: Cardinal read getDecimals write setDecimals;

    //Somewhere we can store temporary data
    property Data: pointer read getData write setData;

    constructor create(init: pUDF_INIT);
    destructor destroy; override;
  end;

implementation

constructor TUDF_Args.create(args: pUDF_ARGS);
var t1: integer;
  arg: TUDF_Argument;
begin
  fargs := TObjectList.Create;
  for t1 := 0 to args.arg_count - 1 do begin
    arg := TUDF_Argument.create(args, t1);
    fargs.add(arg);
  end;
end;

destructor TUDF_Args.destroy;
begin
  fargs.Free;
  inherited;
end;

function TUDF_Args.getargument(index: integer): TUDF_Argument;
begin
  result := TUDF_Argument(fargs[index]);
end;

function TUDF_Args.Count: integer;
begin
  result := fargs.Count;
end;

function TUDF_Argument.getisnull: boolean;
begin
  result := fargs.args[findex] = nil;
end;

function TUDF_Argument.getArgType: ItemResult;
begin
  result := fargs.arg_type[findex];
end;

procedure TUDF_Argument.setArgType(value: ItemResult);
begin
  fargs.arg_type[findex] := value;
end;

function TUDF_Argument.getAttributes: string;
begin
  Result := fargs.attributes[findex];
end;

function TUDF_Argument.getDataInt: int64;
begin
  result := pint64(fargs.args[findex])^;
end;

function TUDF_Argument.getDataDouble: double;
begin
  result := pdouble(fargs.args[findex])^;
end;

function TUDF_Argument.getDataString: string;
begin
  setlength(result, fargs.lengths[findex]);
  move(fargs.args[findex]^, result[1], length(result));
end;

constructor TUDF_Argument.create(args: pUDF_ARGS; index: integer);
begin
  fargs := args;
  findex := index;
end;

function TUDF_Argument.getcanbenull: Boolean;
begin
  result := fargs.maybe_null[findex] <> 0;
end;

procedure TUDF_Argument.setcanbenull(value: boolean);
begin
  if value then
    fargs.maybe_null[findex] := 1 else
    fargs.maybe_null[findex] := 0;
end;

{ TUDF_Init }

constructor TUDF_Init.create(init: pUDF_INIT);
begin
  fInit := init;
end;

destructor TUDF_Init.destroy;
begin

  inherited;
end;

function TUDF_Init.getCanBeNull: boolean;
begin
  result := finit.maybe_null <> 0;
end;

function TUDF_Init.getData: pointer;
begin
  result := finit.ptr;
end;

function TUDF_Init.getDecimals: Cardinal;
begin
  result := finit.decimals;
end;

function TUDF_Init.getMaxStrLen: cardinal;
begin
  result := finit.max_length;
end;

procedure TUDF_Init.setCanBeNull(const Value: boolean);
begin
  if value then
    finit.maybe_null := 1 else
    finit.maybe_null := 0;
end;

procedure TUDF_Init.setData(const Value: pointer);
begin
  finit.ptr := value;
end;

procedure TUDF_Init.setDecimals(const Value: Cardinal);
begin
  fInit.decimals := value;
end;

procedure TUDF_Init.setMaxStrLen(const Value: cardinal);
begin
  fInit.max_length := value;
end;

end.

